package gov.va.genisis2.converter;

import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.convert.converter.Converter;
import org.springframework.stereotype.Component;

import gov.va.genisis2.common.enums.UserStatusEnum;
import gov.va.genisis2.dto.UserDTO;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.model.RoleType;
import gov.va.genisis2.model.User;
import gov.va.genisis2.model.UserApprover;
import gov.va.genisis2.model.UserRoleType;
import gov.va.genisis2.util.DateUtil;
import gov.va.genisis2.vo.LdapUser;

/**
 * This class is used to convert UserDTO to User entity and vice-versa.
 * 
 * @author PII
 *
 */
@Component
public class UserConverter implements Converter<User, UserDTO> {

	private static final Logger LOGGER = LoggerFactory.getLogger(UserConverter.class);
	
	/** The userRoleTypeConverter. */
	@Autowired
	private UserRoleTypeConverter userRoleTypeConverter;

	/** The userApproverConverter. */
	@Autowired
	private UserApproverConverter userApproverConverter;

	@Override
	public UserDTO convert(User user) {
		UserDTO userDto = new UserDTO();

		userDto.setUserId(user.getUserId());
		userDto.setFirstName(user.getFirstName());
		userDto.setMiddleName(user.getMiddleName());
		userDto.setLastName(user.getLastName());
		userDto.setLastName2(user.getLastName2());
		userDto.setCreateDate(user.getCreateDate());
		userDto.setEmailId(user.getEmailId());
		userDto.setUserActive(user.getUserActive());
		userDto.setStatus(UserStatusEnum.getStatus(user.getUserActive()));
		userDto.setActiveDate(user.getActiveDate());
		userDto.setInactiveDate(user.getInactiveDate());
		userDto.setUsername(user.getUsername());

		Set<UserRoleType> userRoleTypes = user.getUserRoleTypes();
		if (!userRoleTypes.isEmpty()) {
			userDto.setUserRoleTypes(userRoleTypeConverter.convertUserRoleTypes(userRoleTypes));
		}

		Set<UserApprover> userApprovers = user.getUserApproversForUserId();
		if (!userApprovers.isEmpty()) {
			userDto.setUserApproversForUserId(userApproverConverter.convertUserApprovers(userApprovers));
		}

		populateRoleDetails(user, userDto);
		userDto.setApprover(this.getUserApprover(user));

		return userDto;
	}
	
	private void populateRoleDetails(User user, UserDTO userDto) {
		try {
			Set<UserRoleType> userRoleTypes = user.getUserRoleTypes();
			if (null == userRoleTypes || userRoleTypes.isEmpty()) {
				LOGGER.error(ErrorEnum.USER_ROLETYPES_IS_NULL.getErrorMessage(), user.getUserId());
				return;
			}
			
			UserRoleType userRoleType = userRoleTypes.iterator().next();
			if (null == userRoleType) {
				LOGGER.error(ErrorEnum.USER_ROLETYPE_IS_NULL.getErrorMessage(), user.getUserId());
				return;
			}
			
			RoleType roleType = userRoleType.getRoleType();
			if (null == roleType) {
				LOGGER.error(ErrorEnum.ROLE_TYPE_IS_NULL.getErrorMessage(), user.getUserId());
				return;
			}

			userDto.setRole((null != roleType.getRoleName()) ? roleType.getRoleName() : null);
			userDto.setRoleDesc((null != roleType.getRoleDescription()) ? roleType.getRoleDescription() : null);
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETROLE.getErrorMessage(), e);
		}
	}

	public String getUserApprover(User user) {
		String approver = null;

		try {
			Set<UserApprover> userApprovers = user.getUserApproversForUserId();
			if (null == userApprovers || userApprovers.isEmpty()) {
				LOGGER.info(ErrorEnum.USERAPPROVERS_IS_NULL.getErrorMessage(), user.getUserId());
				return null;
			}
			UserApprover userApprover = userApprovers.iterator().next();
			if (null == userApprover) {
				LOGGER.error(ErrorEnum.USERAPPROVER_IS_NULL.getErrorMessage(), user.getUserId());
				return null;
			}
			User userByUserApproverId = userApprover.getUsersByUserApproverId();
			approver = (null != userByUserApproverId) ? userByUserApproverId.getUsername() : null;
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETAPPROVER.getErrorMessage(), e);
			return null;
		}

		return approver;
	}
	
	public User convert(LdapUser ldapUser) {
		User user = new User();
		
		user.setFirstName(ldapUser.getFirstName());
		user.setLastName(ldapUser.getLastName());
		
		String userName = ldapUser.getUsername();
		user.setUsername((null == userName || userName.isEmpty()) ? StringUtils.EMPTY : userName.trim().toUpperCase());
		
		String emailId = ldapUser.getEmailId();
		user.setEmailId((null == emailId || emailId.isEmpty()) ? StringUtils.EMPTY : emailId.trim().toUpperCase());
		
		user.setCreateDate(DateUtil.getTodaysDate());
		user.setUserActive((byte) UserStatusEnum.ACTIVE.getId());
		user.setActiveDate(DateUtil.getTodaysDate());
		user.setInactiveDate(ldapUser.getInactiveDate());
		
		return user;
	}
	
	public void populate(User user, LdapUser ldapUser) {
		user.setFirstName(ldapUser.getFirstName());
		user.setLastName(ldapUser.getLastName());
		
		String userName = ldapUser.getUsername();
		user.setUsername((null == userName || userName.isEmpty()) ? StringUtils.EMPTY : userName.toUpperCase());
		
		String emailId = ldapUser.getEmailId();
		user.setEmailId((null == emailId || emailId.isEmpty()) ? StringUtils.EMPTY : emailId.trim().toUpperCase());

		if (user.getUserActive() == UserStatusEnum.DISABLE.getId()) {
			user.setUserActive((byte) UserStatusEnum.ACTIVE.getId());
			user.setActiveDate(DateUtil.getTodaysDate());
		}
		user.setModifiedOn(DateUtil.getTodaysDate());
	}
	
	public void populateDisabledUser(User user) {
		user.setUserActive((byte) UserStatusEnum.DISABLE.getId());
		user.setInactiveDate(DateUtil.getTodaysDate());
		user.setModifiedOn(DateUtil.getTodaysDate());
	}


}